/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

/*
   This module contains the following operators:

*/

#include "cdo_options.h"
#include "process_int.h"
#include "pmlist.h"

static void
dump_cmor_table(const PMList &pmlist)
{
  printf("# Number of lists: %zu\n", pmlist.size());
  int i = 0;
  for (const auto &kvlist : pmlist)
    {
      printf("# list ID: %d;   Number of elements: %zu\n", i, kvlist.size());
      printf("&%s\n", kvlist.name.c_str());
      for (const auto &kv : kvlist) { printf("  %s = %s\n", kv.key.c_str(), kv.values[0].c_str()); }
      printf("/\n");
      ++i;
    }
}

static void
conv_cmor_table(const PMList &pmlist)
{
  const char *hname = "Header";
  const char *vname = "variable";
  // const char *aname = "axis";

  bool hasmissval = false;
  double missval = 0;

  for (const auto &kvlist : pmlist)
    {
      const auto listname = kvlist.name.c_str();

      if (std::strncmp(listname, hname, strlen(hname)) == 0)
        {
          for (const auto &kv : kvlist)
            {
              const auto ename = kv.key.c_str();
              const auto evalue = kv.values[0].c_str();
              const size_t len = strlen(ename);

              if (std::strncmp("missing_value", ename, len) == 0)
                {
                  missval = atof(evalue);
                  hasmissval = true;
                }
            }
        }
      else if (std::strncmp(listname, vname, strlen(vname)) == 0)
        {
          printf("&%s\n", "parameter");
          for (const auto &kv : kvlist)
            {
              const char *ename = kv.key.c_str();
              const char *evalue = kv.values[0].c_str();
              const int len = strlen(ename);
              int vlen = strlen(evalue);

              if (vlen > 1 && evalue[0] == '"' && evalue[vlen - 1] == '"')
                {
                  vlen -= 2;
                  evalue++;
                }

              char *ovalue = strdup(evalue);
              for (int i = 1; i < vlen; ++i)
                {
                  if (ovalue[i - 1] == '"' && ovalue[i] == '"')
                    {
                      ovalue[i - 1] = '\'';
                      for (int j = i + 1; j < vlen; ++j) ovalue[j - 1] = ovalue[j];
                      vlen -= 1;
                    }
                }

              if (vlen)
                {
                  if (std::strncmp("name", ename, len) == 0 || std::strncmp("standard_name", ename, len) == 0
                      || std::strncmp("out_name", ename, len) == 0 || std::strncmp("type", ename, len) == 0
                      || std::strncmp("valid_min", ename, len) == 0 || std::strncmp("valid_max", ename, len) == 0
                      || std::strncmp("ok_min_mean_abs", ename, len) == 0 || std::strncmp("ok_max_mean_abs", ename, len) == 0)
                    printf("  %-15s = %s\n", ename, ovalue);
                  else if (std::strncmp("long_name", ename, len) == 0 || std::strncmp("units", ename, len) == 0
                           || std::strncmp("cell_methods", ename, len) == 0 || std::strncmp("cell_measures", ename, len) == 0
                           || std::strncmp("comment", ename, len) == 0)
                    printf("  %-15s = \"%.*s\"\n", ename, vlen, ovalue);
                }

              free(ovalue);
            }
          if (hasmissval) printf("  %-15s = %g\n", "missing_value", missval);
          printf("/\n");
        }
    }
}

class CMOR_table : public Process
{
public:
  using Process::Process;
  inline static CdoModule module = {
    .name = "CMOR_table",
    .operators = { { "dump_cmor_table"}, { "conv_cmor_table"} },
    .aliases = {},
    .mode = EXPOSED,     // Module mode: 0:intern 1:extern
    .number = CDI_REAL,  // Allowed number type
    .constraints = { 0, 0, NoRestriction },
  };
  inline static RegisterEntry<CMOR_table> registration = RegisterEntry<CMOR_table>(module);
  int DUMP_CMOR_TABLE, CONV_CMOR_TABLE;
  FILE *fp;
  const char *filename;
  int operatorID;

public:
  void
  init()
  {

    DUMP_CMOR_TABLE = module.get_id("dump_cmor_table");
    CONV_CMOR_TABLE = module.get_id("conv_cmor_table");

    operatorID = cdo_operator_id();

    if (cdo_operator_argc() != 1) cdo_abort("Too few arguments!");
    filename = cdo_operator_argv(0).c_str();

    if (Options::cdoVerbose) cdo_print("Parse file: %s", filename);

    fp = std::fopen(filename, "r");
    if (fp == nullptr) cdo_abort("Open failed on: %s\n", filename);
  }
  void
  run()
  {
    PMList pmlist;
    pmlist.read_cmor_table(fp, filename);
    std::fclose(fp);

    if (operatorID == DUMP_CMOR_TABLE)
      dump_cmor_table(pmlist);
    else if (operatorID == CONV_CMOR_TABLE)
      conv_cmor_table(pmlist);
  }
  void
  close()
  {
  }
};
