/*
 * Copyright (C) 2007-2009 KenD00
 * 
 * This file is part of DumpHD.
 * 
 * DumpHD is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package dumphd.aacs;

import dumphd.core.KeyData;
import dumphd.util.MessagePrinter;


/**
 * This is a wrapper class to access the platform dependent AACSKeys library.
 * 
 * The library gets loaded when this class gets instantiated. Its methods map to library methods.
 * 
 * @author KenD00
 */
public class AACSKeys {

   /**
    * MessagePrinter passed to the library for textual output
    */
   private MessagePrinter mp = null;


   /**
    * Constructs an AACSKeys object and loads the AACSKeys library.
    * 
    * @param mp The MessagePrinter used for textual output by the library
    * @throws AACSException An error occurred loading the library
    */
   public AACSKeys(MessagePrinter mp) throws AACSException {
      this.mp = mp;
      try {
         System.loadLibrary("aacskeys");
      }
      catch (NullPointerException e1) {
         // This can never happen, but rethrow the exception anyway
         throw new AACSException(e1.getMessage(), e1);
      }
      catch (SecurityException e2) {
         throw new AACSException(e2.getMessage(), e2);
      }
      catch (UnsatisfiedLinkError e3) {
         // This exception will be most frequently raised, it does not only mean that the library has not been found,
         // there may also be another problem with the library!
         throw new AACSException(e3.getMessage(), e3);
      }
   }

   /**
    * Returns the version of the library.
    * 
    * @return A human readable string of the library version
    */
   public native String getVersionString();

   /**
    * Retrieves the AACS keys for the given KeyData object.
    * 
    * The keys are directly added to the object.
    * 
    * @param sourcePath The path to the source. This string must contain the platform specific path to the source
    * @param kd The KeyData object the keys get retrieved for
    * @throws AACSException An error occurred retrieving the keys, however some keys may have been retrieved
    */
   public native void getKeys(String sourcePath, KeyData kd) throws AACSException;

}
