% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assert.R
\name{vector-checks}
\alias{vector-checks}
\alias{obj_is_vector}
\alias{obj_check_vector}
\alias{vec_check_size}
\title{Vector checks}
\usage{
obj_is_vector(x)

obj_check_vector(x, ..., arg = caller_arg(x), call = caller_env())

vec_check_size(x, size, ..., arg = caller_arg(x), call = caller_env())
}
\arguments{
\item{x}{For \verb{obj_*()} functions, an object. For \verb{vec_*()} functions, a
vector.}

\item{...}{These dots are for future extensions and must be empty.}

\item{arg}{An argument name as a string. This argument
will be mentioned in error messages as the input that is at the
origin of a problem.}

\item{call}{The execution environment of a currently
running function, e.g. \code{caller_env()}. The function will be
mentioned in error messages as the source of the error. See the
\code{call} argument of \code{\link[rlang:abort]{abort()}} for more information.}

\item{size}{The size to check for.}
}
\value{
\itemize{
\item \code{obj_is_vector()} returns a single \code{TRUE} or \code{FALSE}.
\item \code{obj_check_vector()} returns \code{NULL} invisibly, or errors.
\item \code{vec_check_size()} returns \code{NULL} invisibly, or errors.
}
}
\description{
\itemize{
\item \code{obj_is_vector()} tests if \code{x} is considered a vector in the vctrs sense.
See \emph{Vectors and scalars} below for the exact details.
\item \code{obj_check_vector()} uses \code{obj_is_vector()} and throws a standardized and
informative error if it returns \code{FALSE}.
\item \code{vec_check_size()} tests if \code{x} has size \code{size}, and throws an informative
error if it doesn't.
}
}
\section{Vectors and scalars}{


Informally, a vector is a collection that makes sense to use as column in a
data frame. The following rules define whether or not \code{x} is considered a
vector.

If no \code{\link[=vec_proxy]{vec_proxy()}} method has been registered, \code{x} is a vector if:
\itemize{
\item The \link[=typeof]{base type} of the object is atomic: \code{"logical"}, \code{"integer"},
\code{"double"}, \code{"complex"}, \code{"character"}, or \code{"raw"}.
\item \code{x} is a list, as defined by \code{\link[=obj_is_list]{obj_is_list()}}.
\item \code{x} is a \link{data.frame}.
}

If a \code{vec_proxy()} method has been registered, \code{x} is a vector if:
\itemize{
\item The proxy satisfies one of the above conditions.
\item The base type of the proxy is \code{"list"}, regardless of its class. S3 lists
are thus treated as scalars unless they implement a \code{vec_proxy()} method.
}

Otherwise an object is treated as scalar and cannot be used as a vector.
In particular:
\itemize{
\item \code{NULL} is not a vector.
\item S3 lists like \code{lm} objects are treated as scalars by default.
\item Objects of type \link{expression} are not treated as vectors.
}
}

\section{Technical limitations}{

\itemize{
\item Support for S4 vectors is currently limited to objects that inherit from an
atomic type.
\item Subclasses of \link{data.frame} that \emph{append} their class to the back of the
\code{"class"} attribute are not treated as vectors. If you inherit from an S3
class, always prepend your class to the front of the \code{"class"} attribute
for correct dispatch. This matches our general principle of allowing
subclasses but not mixins.
}
}

\examples{
obj_is_vector(1)

# Data frames are vectors
obj_is_vector(data_frame())

# Bare lists are vectors
obj_is_vector(list())

# S3 lists are vectors if they explicitly inherit from `"list"`
x <- structure(list(), class = c("my_list", "list"))
obj_is_list(x)
obj_is_vector(x)

# But if they don't explicitly inherit from `"list"`, they aren't
# automatically considered to be vectors. Instead, vctrs considers this
# to be a scalar object, like a linear model returned from `lm()`.
y <- structure(list(), class = "my_list")
obj_is_list(y)
obj_is_vector(y)

# `obj_check_vector()` throws an informative error if the input
# isn't a vector
try(obj_check_vector(y))

# `vec_check_size()` throws an informative error if the size of the
# input doesn't match `size`
vec_check_size(1:5, size = 5)
try(vec_check_size(1:5, size = 4))
}
