/* $Id: errno.c,v 1.6 2002/03/02 19:37:36 sverrehu Exp $ */
/*------------------------------------------------------------------------
 |  FILE            errno.c
 |  MODULE OF       shhmsg - library for displaying messages.
 |
 |  DESCRIPTION     The functions in this file, are for displaying errno-
 |                  dependant error messages.
 |
 |  WRITTEN BY      Sverre H. Huseby <shh@thathost.com>
 +----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

#include "internal.h"
#include "shhmsg.h"

#ifndef HAVE_STRERROR
extern const char *const sys_errlist[];
#define strerror(x) sys_errlist[x]
#endif

/*-----------------------------------------------------------------------+
|  PUBLIC FUNCTIONS                                                      |
+-----------------------------------------------------------------------*/

/*------------------------------------------------------------------------
 |
 |  NAME          msgPerror
 |
 |  FUNCTION      Show an errno-dependant error message.
 |
 |  SYNOPSIS      #include "shhmsg.h"
 |                void msgPerror(const char *format, ...);
 |
 |  INPUT         format, ...
 |                        Arguments used as with printf(). If NULL,
 |                        the name of the program is printed.
 |
 |  DESCRIPTION   Prints the given text followed by ": " and the
 |                correct errormessage on the _msgErrorStream.
 |		  May add the program name before anything is printed.
 |
 |                If errno indicates "no error", nothing is printed.
 */
void
msgPerror(const char *format, ...)
{
    va_list ap;
    int en;

    if (!(en = errno))
        return;
    fflush(stdout);
    if (format && *format) {
	if (_msgShowNameAlways)
	    fprintf(GET_ERROR_STREAM, "%s: ", msgGetName());
	va_start(ap, format);
	vfprintf(GET_ERROR_STREAM, format, ap);
	va_end(ap);
    } else
	fputs(msgGetName(), GET_ERROR_STREAM);
    fprintf(GET_ERROR_STREAM, ": %s\n", strerror(en));
}

/*------------------------------------------------------------------------
 |
 |  NAME          msgFatalPerror
 |
 |  FUNCTION      Show an errno-dependant error message and abort the program.
 |
 |  SYNOPSIS      #include "shhmsg.h"
 |                void msgFatalPerror(const char *format, ...);
 |
 |  INPUT         format, ...
 |                        Arguments used as with printf(). If NULL,
 |                        the name of the program is printed.
 |
 |  RETURNS       Nothing, never returns.
 |
 |  DESCRIPTION   Prints the given text followed by ": " and the
 |                correct errormessage on the _msgErrorStream, and then
 |                aborts the program.
 |		  May add the program name before anything is printed.
 |
 |                If errno indicates "no error", nothing is printed,
 |                and the program keeps running.
 */
void
msgFatalPerror(const char *format, ...)
{
    va_list ap;
    int en;

    if (!(en = errno))
        return;
    fflush(stdout);
    if (format && *format) {
	if (_msgShowNameAlways)
	    fprintf(GET_ERROR_STREAM, "%s: ", msgGetName());
	va_start(ap, format);
	vfprintf(GET_ERROR_STREAM, format, ap);
	va_end(ap);
    } else
	fputs(msgGetName(), GET_ERROR_STREAM);
    fprintf(GET_ERROR_STREAM, ": %s\n", strerror(en));
    exit(99);
}
