use ui;
use core:io;

dialog Dialog {
	layout Grid {
		expandCol: 0;

		Label status("Looking for errors...") { colspan: 2; }
		nextLine;

		ui:Progress progress() { colspan: 2; }
		nextLine;

		Label message("") { colspan: 2; }
		FillBox(300, 10) { row: 2; col: 0; }
		nextLine;

		Button ok("Cancel") { col: 1; }
	}

	init(Url[] files) {
		init("Looking for concurrency errors...") {
			files = files;
		}

		defaultChoice = ok;

		progress.wait;

		spawn check();
	}

	// Cancel our progress?
	private Bool cancel;

	// Are we done?
	private Bool done;

	// Result, if any.
	Result? result;

	// Update status. Returns 'false' if we shall abort search.
	private Bool update(Nat states, Nat edges, Nat unexplored) {
		message.text = "Explored ${states} states and ${edges} transitions...";

		return !cancel;
	}

	// Window closed. Abort any running task.
	void windowDestroyed() {
		cancel = true;
	}

	// Called when we are done.
	void onDestroy(Int code) : override {
		cancel = true;
	}

	// Set us into an error state.
	void error(Str message) {
		progress.progress = 1.0;
		status.text = "Search failed.";
		ok.text = "Close";
		this.message.text = "An internal error occurred: " + message;

		size = minSize();
	}

	// Set us in a "done" state.
	void done(Result? result) {
		this.result = result;
		progress.progress = 1.0;
		status.text = "Done!";
		if (result) {
			message.text = "Found an error of type: ${result.error.type}\nClick \"Show\" below to see the issue.";
			ok.text = "Show";
		} else {
			message.text = "No errors found, at least not with this main program.";
			ok.text = "Close";
		}

		size = minSize();
	}

	private Url[] files;

	private void check() {
		try {
			var result = check(files, UiProgress(this));
			done(result);
		} catch (progvis:program:ExitError e) {
			// Early exit.
		} catch (Exception e) {
			error(e.message);
		}
	}

	// Progress callbacks.
	class UiProgress extends Progress {
		init(Dialog owner) {
			init { owner = owner; }
		}

		Dialog owner;

		Moment lastCheck;

		void progress(Nat depth, Nat states, Nat edges, Nat unexplored) : override {
			Moment now;
			if (now - lastCheck > 1 s) {
				lastCheck = now;

				if (!owner.update(states, edges, unexplored))
					throw progvis:program:ExitError();
			}
		}
	}
}


// Show a dialog for finding errors, and return the result.
Result? findErrorsUi(Frame parent, Url[] files) on Ui {
	Dialog d(files);
	if (d.show(parent) > 0) {
		return d.result;
	}
	null;
}
