use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * A column declaration.
 */
class Column {
	// Name of the column.
    Str name;

	// Datatype.
	SQLType datatype;

	// Is this column a primary key?
	Bool primaryKey;

	// Is this column allowed to contain NULL?
	Bool allowNull;

	// Unique?
	Bool unique;

	// Auto-increment?
	Bool autoIncrement;

	// Default value?
	SQLLiteral? default;


	init(SStr name, SQLType type) {
		init {
			name = name.v;
			datatype = type;
		}
	}

	// Called from the syntax.
	void setPrimary(Str x) { primaryKey = true; }
	void setAllowNull(Str x) { allowNull = true; }
	void setUnique(Str x) { unique = true; }
	void setAutoIncrement(Str x) { autoIncrement = true; }
	void setDefault(SQLLiteral x) { default = x; }

	// Called from the syntax, gives a decent error message on how we're different from SQL.
	void setNotNull(SStr x) {
		throw SyntaxError(x.pos, "Columns are NOT NULL by default. Use ALLOW NULL to allow nullable columns.");
	}

	// Create an SQL part for this column.
	void toSQL(QueryStrBuilder to) {
		to.name(name);
		to.put(" ");
		to.type(datatype);
		modifiers(to);
	}

	void toS(StrBuf to) : override {
		QueryStrBuilder b;
		toSQL(b);
		to << b.build;
		if (primaryKey)
			to << " (primary key)";
	}

	// Check if we are similar enough to "other".
	Bool sameAs(Schema:Column other) {
		Bool same = name == other.name;
		same &= datatype.sqlType.compatible(other.type);
		same &= primaryKey == other.attributes.has(Schema:Attributes:primaryKey);
		same &= allowNull != other.attributes.has(Schema:Attributes:notNull);
		same &= unique == other.attributes.has(Schema:Attributes:unique);
		same &= autoIncrement == other.attributes.has(Schema:Attributes:autoIncrement);

		if (default) {
			if (odefault = other.default) {
				same &= default.same(odefault);
			} else {
				same = false;
			}
		} else {
			// It is fine if the DB has a default value that we don't know about. We will not
			// utilize it anyway.
		}

		same;
	}

	// See if this column has some kind of default value. I.e. it is OK to leave one without a value?
	Bool hasDefault(Bool multiPK) {
		if (default)
			return true;
		if (primaryKey & !multiPK)
			return true;
		if (allowNull)
			return true;
		if (autoIncrement)
			return true;
		false;
	}

	private void modifiers(QueryStrBuilder to) {
		if (!allowNull) {
			to.put(" NOT NULL");
		}
		if (unique) {
			to.put(" UNIQUE");
		}
		if (autoIncrement) {
			to.put(" ");
			to.autoIncrement();
		}
		if (default) {
			to.put(" DEFAULT " + default.toSQL());
		}
	}
}


/**
 * Index for a table.
 */
class Index {
	// Name of the index (might be auto-generated).
	Str name;

	// Columns.
	Str[] columns;

	// Create.
	init(Str name, Str[] columns) {
		init {
			name = name;
			columns = columns;
		}
	}

	// To SQL statement.
	void toSQL(QueryStrBuilder to, Str table) {
		to.put("CREATE INDEX ");
		to.name(name);
		to.put(" ON ");
		to.name(table);
		to.put("(");
		for (i, c in columns) {
			if (i > 0)
				to.put(", ");
			to.name(c);
		}
		to.put(");");
	}

	// To string. No SQL escaping.
	void toS(StrBuf to) : override {
		to << "INDEX " << name << " ON ?(" << join(columns, ", ") << ")";
	}
}

/**
 * Declaration of an entire table.
 */
class Table {
	// Name of the table.
    Str name;

	// Columns in the table.
    Column[] columns;

	// Indices for this table.
	Index[] indices;

	init(SStr name) {
		init {
			name = name.v;
		}
	}

	// Find a column.
	Column? find(Str name) {
		for (c in columns)
			if (c.name == name)
				return c;
		null;
	}

	// Add column (called from syntax).
	void add(Column col) {
		columns << col;
	}

	// Add primary key(s) (called from syntax).
	void add(Array<SStr> cols) {
		for (c in columns)
			if (c.primaryKey)
				throw SyntaxError(cols[0].pos, "Only one instance of the PRIMARY KEY keyword may be present for each table.");

		for (c in cols) {
			unless (col = find(c.v))
				throw SyntaxError(c.pos, "No column named ${c.v} was declared in this table.");
			col.primaryKey = true;
		}
	}

	// Add index.
	void add(SrcPos pos, Index index) {
		for (i in indices)
			if (i.name == index.name)
				throw SyntaxError(pos, "The index ${index.name} already exists.");

		indices << index;
	}

	// Check if there are multiple primary keys.
	Bool multiplePK() {
		Nat count = 0;
		for (c in columns)
			if (c.primaryKey)
				count++;
		count > 1;
	}

	// Figure out what index, if any, has an implicit AUTOINCREMENT. Returns "count" if none has it.
	Nat implicitAutoIncrementColumn() {
		Nat count = columns.count();
		Nat firstPrimary = count();

		for (i, c in columns) {
			if (c.primaryKey) {
				if (firstPrimary < count)
					return count;
				firstPrimary = i;
			}
		}

		var col = columns[firstPrimary];
		// Reset if it is either already marked as autoincrement, or if it is of the improper type.
		if (col.autoIncrement) {
			firstPrimary = count;
		} else if (!col.datatype.sqlType.sameType(QueryType:integer)) {
			firstPrimary = count;
		}

		return firstPrimary;
	}

	// Create an SQL statement for this table declaration.
	void toSQL(QueryStrBuilder to, Bool ifNotExists, Bool implicitAutoIncrement, Str[] options) {
		to.put("CREATE TABLE ");
		if (ifNotExists)
			to.put("IF NOT EXISTS ");
		to.name(name);
		to.put(" (");

		Str[] pk;
		for (col in columns)
			if (col.primaryKey)
				pk << col.name;

		Nat implicitCol = if (implicitAutoIncrement) { columns.count; } else { implicitAutoIncrementColumn(); };
		for (i, col in columns) {
			if (i > 0)
				to.put(", ");
			col.toSQL(to);

			// Note: Some databases (e.g. SQLite) require that the PRIMARY KEY is specified here in
			// order for AUTOINCREMENT to work properly.
			if (pk.count == 1 & col.primaryKey)
				to.put(" PRIMARY KEY");

			if (implicitCol == i) {
				to.put(" ");
				to.autoIncrement();
			}
		}

		if (pk.count > 1) {
			to.put(", PRIMARY KEY(");
			for (i, k in pk) {
				if (i > 0)
					to.put(", ");
				to.name(k);
			}
			to.put(")");
		}

		to.put(")");
		if (options.any) {
			to.put(" ");
			for (i, k in options) {
				if (i > 0)
					to.put(", ");
				to.put(k);
			}
		}
		to.put(";");
	}
}

/**
 * Helper type for creating indices.
 */
class IndexDecl {
	SrcPos pos;
	SStr table;
	Index index;

	init(SrcPos pos, SStr name, SStr table, SStr[] cols) {
		Str[] c;
		for (x in cols)
			c << x.v;

		init {
			pos = pos;
			table = table;
			index(name.v, c);
		}
	}

	init(SrcPos pos, SStr table, SStr[] cols) {
		StrBuf name;
		name << table.v;

		Str[] c;
		for (x in cols) {
			c << x.v;
			name << "_" << x.v;
		}

		init {
			pos = pos;
			table = table;
			index(name.toS, c);
		}
	}
}

/**
 * Database description.
 */
class Database {
	// Tables declared (indices are stored inside each table).
	Table[] tables;

	// Add a table.
	void add(Table decl) {
		tables.push(decl);
	}

	// Add an index.
	void add(IndexDecl decl) {
		unless (table = find(decl.table.v))
			throw SyntaxError(decl.table.pos, "The table ${decl.table.v} was not declared (yet).");

		table.add(decl.pos, decl.index);
	}

	// Find a table.
	Table? find(Str name) {
		// TODO: Speedier lookup?
		for (table in tables)
			if (table.name == name)
				return table;
		null;
	}
}

/**
 * Declaration of a database.
 */
class DatabaseDecl extends NamedDecl {
    SStr name;
    Scope scope;
	Database contents;

    init(SStr name, Scope scope, Database contents) {
        init() {
            name = name;
            scope = scope;
			contents = contents;
        }
    }

    Named doCreate() {
        DatabaseType(name, contents);
    }
}

/**
 * Type stored in the name tree.
 */
class DatabaseType extends Type {
    Database contents;

    init(SStr name, Database contents) {
        init(name.v, TypeFlags:typeClass) {
            contents = contents;
        }

		setSuper(named{TypedConnection});
		addCtor();
    }

	private void addCtor() {
		BSTreeCtor ctor([thisParam(this), ValParam(named{DBConnection}, "db")], SrcPos());
		CtorBody body(ctor, Scope());
		if (var = body.variable(SimplePart("db"))) {
			Actuals params;
			params.add(LocalVarAccess(SrcPos(), var));
			params.add(ReferDBType(this));
			body.add(InitBlock(SrcPos(), body, params));
		} else {
			throw InternalError("Unable to find the 'db' parameter.");
		}

		ctor.body = body;
		add(ctor);
	}
}

/**
 * Expression that returns a type.
 */
private class ReferDBType extends Expr {
	private DatabaseType referTo;

	init(DatabaseType referTo) {
		init(SrcPos()) { referTo = referTo; }
	}

	ExprResult result() : override {
		ExprResult(Value(named{DatabaseType}));
	}

	void code(CodeGen state, CodeResult result) : override {
		if (result.needed) {
			var v = result.location(state);
			state.l << core:asm:mov(v, core:asm:objPtr(referTo));
			result.created(state);
		}
	}

	Str toS() : override {
		"<type>";
	}
}

/**
 * Base class inherited from the DBType class.
 */
class TypedConnection {
	// Underlying database connection.
	DBConnection connection;

	// Cached prepared statements. TODO: Copy these on a thread switch?
	private Nat->Statement statements;

	// Create and verify the database structure.
	init(DBConnection conn, DatabaseType t) {
		init { connection = conn; }

		checkStructure(t.contents);
	}

	// Close.
	void close() {
		for (k, v in statements)
			v.finalize();
		connection.close();
	}

	// Get a cached prepared statement based on its ID.
	// TODO: One idea here is to remove statements that are "prepared" and not return them
	// until we are done with them. That allows us to be re-entrant properly.
	Statement prepare(Nat id, QueryStr query) {
		var iter = statements.find(id);
		if (iter == statements.end) {
			Statement stmt = connection.prepare(query);
			statements.put(id, stmt);
			return stmt;
		} else {
			return iter.v;
		}
	}

	// Check the database contents for a match.
	private void checkStructure(Database db) {
		// TODO: This should be rewritten and improved.
		for (table in db.tables) {
			if (schema = connection.schema(table.name)) {
				var queries = checkTable(table, schema);
				for (q in queries) {
					var stmt = connection.prepare(q);
					stmt.execute();
					stmt.finalize();
				}
			} else {
				// Create the table.
				Str[] options;
				// TODO: This should be handled in a more general way...
				if (connection as SQLite)
					options << "STRICT";
				var queryStr = {
					QueryStrBuilder builder;
					table.toSQL(builder, false, connection.implicitAutoIncrement, options);
					builder.build();
				};
				var stmt = connection.prepare(queryStr);
				stmt.execute();
				stmt.finalize();

				for (index in table.indices) {
					var queryStr = {
						QueryStrBuilder builder;
						index.toSQL(builder, table.name);
						builder.build();
					};
					var stmt = connection.prepare(queryStr);
					stmt.execute();
					stmt.finalize();
				}
			}
		}
	}

	// Check a single table, and perform migrations as required.
	private QueryStr[] checkTable(Table table, Schema schema) {
		if (table.columns.count < schema.count)
			throw SchemaError("Too many columns in the database.", table, schema);

		QueryStr[] queries;

		if (!connection.implicitAutoIncrement()) {
			// Make changes to a copy of "table" to give better error messages:
			Nat col = table.implicitAutoIncrementColumn();
			if (col < table.columns.count) {
				table = table.clone();
				table.columns[col].autoIncrement = true;
			}
		}

		// Check columns:
		for (Nat i = 0; i < schema.count; i++) {
			var current = schema[i];
			var desired = table.columns[i];

			if (!desired.sameAs(current)) {
				if (current.name == desired.name) {
					if (!connection.fullAlterTable())
						throw SchemaError("Unable to update column ${i} since the database does not support ALTER TABLE fully.", table, schema);

					// This is fairly easy, we can just ask the DB to modify the properties. It
					// might fail though.
					queries << updateColumn(table.name, desired, current);
				} else {
					// This could be done. We could try to find another column with the same name
					// and use that. Renames makes this a bit more difficult, so we don't support it
					// for now.
					throw SchemaError("Incorrect name for column ${i}. New columns can only be added last for auto-migration to work.", table, schema);
				}
			}
		}

		// Add any new columns:
		for (Nat i = schema.count; i < table.columns.count; i++) {
			var col = table.columns[i];
			QueryStrBuilder to;
			to.put("ALTER TABLE ");
			to.name(table.name);
			to.put(" ADD ");
			// We can't add autoincrement before adding it as a primary key.
			if (col.autoIncrement) {
				col.autoIncrement = false;
				col.toSQL(to);
				col.autoIncrement = true;
			} else {
				col.toSQL(to);
			}
			to.put(";");
			queries << to.build();
		}

		// Check if the set of primary keys have changed. If so, we need to re-create the set of
		// primary keys.
		{
			Bool updatePK = false;
			Bool anyCurrentPK = false;
			Str[] desiredPK;
			for (i, col in table.columns) {
				Bool desired = col.primaryKey;
				if (desired)
					desiredPK << col.name;

				if (i < schema.count) {
					Bool current = schema[i].attributes.has(Schema:Attributes:primaryKey);
					anyCurrentPK |= current;
					updatePK |= (desired != current);
				} else {
					updatePK |= desired;
				}
			}

			if (updatePK) {
				QueryStrBuilder to;
				if (anyCurrentPK) {
					to.put("ALTER TABLE ");
					to.put(table.name);
					to.put(" DROP PRIMARY KEY;");
					queries << to.build;
					to.clear();
				}

				to.put("ALTER TABLE ");
				to.put(table.name);
				to.put(" ADD PRIMARY KEY (");
				for (i, v in desiredPK) {
					if (i > 0)
						to.put(", ");
					to.name(v);
				}
				to.put(");");
				queries << to.build;

				// Check if any of the primary keys need to have the autoincrement keyword added now.
				for (i, col in table.columns) {
					Bool update = col.autoIncrement;
					if (i < schema.count)
						update &= !schema[i].attributes.has(Schema:Attributes:autoIncrement);

					if (update) {
						QueryStrBuilder to;
						to.put("ALTER TABLE ");
						to.put(table.name);
						to.put(" MODIFY ");
						col.toSQL(to);
						queries << to.build;
					}
				}
			}
		}

		// Check indices:
		{
			Set<Str> desired;
			for (x in table.indices)
				desired.put(hashIndex(x));
			Set<Str> current;
			for (x in schema.indices)
				current.put(hashIndex(x));

			// See what to remove:
			for (x in schema.indices) {
				if (!desired.has(hashIndex(x))) {
					QueryStrBuilder to;
					to.put("DROP INDEX ");
					to.name(x.name);
					to.put(";");
					queries << to.build();
				}
			}

			// See what to add:
			for (x in table.indices) {
				if (!current.has(hashIndex(x))) {
					QueryStrBuilder to;
					x.toSQL(to, table.name);
					queries << to.build();
				}
			}
		}

		queries;
	}

	QueryStr updateColumn(Str table, Column desired, Schema:Column current) {
		QueryStrBuilder to;
		to.put("ALTER TABLE ");
		to.name(table);
		if (desired.name == current.name) {
			to.put(" MODIFY ");
		} else {
			to.put(" CHANGE ");
			to.name(current.name);
			to.put(" ");
		}

		if (!current.attributes.has(Schema:Attributes:primaryKey) & desired.autoIncrement) {
			// We can't add autoincrement before setting it as a primary key. So, temporarily ignore
			// the autoincrement part:
			desired.autoIncrement = false;
			desired.toSQL(to);
			desired.autoIncrement = true;
		} else {
			desired.toSQL(to);
		}
		to.put(";");
		to.build();
	}
}

// Produce a "hash" of an index.
private Str hashIndex(Index i) {
	StrBuf b;
	b << i.name << "|" << join(i.columns, ",");
	b.toS;
}
private Str hashIndex(Schema:Index i) {
	StrBuf b;
	b << i.name << "|" << join(i.columns, ",");
	b.toS;
}
